unit rawbufferUnit;

{* Delphi Streaming Radio Library
 * Copyright 2004-2007, Steve Blinch
 * http://code.blitzaffe.com
 * ============================================================================
 *
 * LICENSE
 *
 * This code is free software; you can redistribute it and/or modify it under the
 * terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *	
 * You should have received a copy of the GNU General Public License along
 * with this code; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *}

interface

const
  MSG_RAWBUF_ALLOC      = 5050;
  MSG_RAWBUF_DEALLOC    = 5051;
  MSG_RAWBUF_NAME       = 5052;

type
//  TRawBuffer = class;
  TRawBuffer = class
    private
      FBuffer: Pointer;
      FSize: Integer;
      FAllocated: Boolean;
      FPeer: TObject;
      FCursor: Integer;
      {$IFDEF RAWBUF_DEBUG}FID: String;{$ENDIF}

      // internal method called by the remote object during a MigrateTo operation
      function ReceiveMigration(Buffer: Pointer; Size: Integer): boolean;
    public

      constructor Create(APeer: TObject); overload;
      constructor Create(Size: Integer); overload;
      constructor Create(Size: Integer; APeer: TObject); overload;
      destructor Destroy; override;

      {$IFDEF RAWBUF_DEBUG}
      procedure SetID(ID: String);
      {$ENDIF}

      // allocates the requested number of bytes; if a buffer is already
      // allocated it will be destroyed first
      function Allocate(Bytes: Integer): Boolean;

      // frees all memory associated with this buffer
      procedure Deallocate;

      // stores Bytes bytes of Data in this buffer; returns TRUE
      // on success, FALSE if not enough room
      function Store(var Data; Bytes: Integer): boolean;

      // retrieves up to Bytes bytes of data from this buffer; returns
      // the number of bytes fetched
      function Fetch(var Container; Bytes: Integer): integer;

      // appends the contents of Buffer to this buffer
      function Append(var Buffer: TRawBuffer): Boolean;

      // removes the specified number of bytes from the beginning of the buffer
      // (thereby decreasing the size of the buffer by that number of bytes)
      procedure RemoveLeadingBytes(Bytes: Integer);

      // causes NewBuffer to take on the data pointer from this buffer;
      // this buffer is then marked as deallocated; if NewBuffer was
      // previously allocated, it is deallocated before migration
      function MigrateTo(var NewBuffer: TRawBuffer): Boolean;

      // cursor functionality allows multiple blocks to be written (or read)
      // consecutively to/from a buffer
      //
      // resets the cursor to the beginning of the buffer
      procedure ResetCursor;
      // stores some data at the cursor position; returns TRUE on success, FALSE
      // if there was not enough room
      function StoreAtCursor(var Data; Bytes: Integer): boolean;
      // stores some data at the cursor position; returns the number of bytes
      // fetched into Container
      function FetchAtCursor(var Container; Bytes: Integer): Integer;

      // creates a new TRawBuffer instance containing a copy of the data in
      // this buffer
      function Copy: TRawBuffer;

      // returns the contents of the buffer as a string
      function AsString: String;
      function StoreString(S: String): boolean;
      function StoreStringAtCursor(S: String): boolean;

      property Buffer: Pointer read FBuffer;
      property Peer: TObject read FPeer;
      property Size: Integer read FSize;
      property Allocated: Boolean read FAllocated;
      {$IFDEF RAWBUF_DEBUG}
      property ID: String read FID;
      {$ENDIF}
    end;

implementation

uses mainUnit,Windows,Messages;

constructor TRawBuffer.Create(Size: Integer; APeer: TObject);
begin
  FAllocated:=False;
  FCursor:=0;
  FPeer:=APeer;
  Allocate(Size);

  {$IFDEF RAWBUF_DEBUG}
  FID:='';
  SendMessage(HWND(frmMain.Handle),WM_USER,MSG_RAWBUF_ALLOC,Integer(Self));
  {$ENDIF}
end;

constructor TRawBuffer.Create(Size: Integer);
begin
  FCursor:=0;
  FAllocated:=False;
  FPeer:=nil;
  Allocate(Size);

  {$IFDEF RAWBUF_DEBUG}
  FID:='';
  SendMessage(HWND(frmMain.Handle),WM_USER,MSG_RAWBUF_ALLOC,Integer(Self));
  {$ENDIF}
end;

constructor TRawBuffer.Create(APeer: TObject);
begin
  FCursor:=0;
  FAllocated:=False;
  FPeer:=APeer;

  {$IFDEF RAWBUF_DEBUG}
  FID:='';
  SendMessage(HWND(frmMain.Handle),WM_USER,MSG_RAWBUF_ALLOC,Integer(Self));
  {$ENDIF}
end;

function TRawBuffer.Allocate(Bytes: Integer): Boolean;
begin
  Result:=False;
  Deallocate;

  if (Bytes<=0) then exit;

  FSize:=Bytes;
  try
    GetMem(FBuffer,FSize);
    FAllocated:=True;
  except
  end;

  Result:=FAllocated;
end;

procedure TRawBuffer.Deallocate;
begin
  if (FAllocated and Assigned(FBuffer)) then
    try
      FreeMem(FBuffer,FSize);
    except
    end;

  FCursor:=0;
  FSize:=0;
  FBuffer:=nil;
  FAllocated:=False;
end;

destructor TRawBuffer.Destroy;
begin
  Deallocate;

  {$IFDEF RAWBUF_DEBUG}PostMessage(HWND(frmMain.Handle),WM_USER,MSG_RAWBUF_DEALLOC,Integer(Self));{$ENDIF}

  inherited;
end;

function TRawBuffer.Store(var Data; Bytes: Integer): boolean;
begin
  if (not FAllocated) then
    begin
      Result:=False;
      exit;
    end;

  // can't store data larger than our buffer
  if (Bytes>FSize) then
    begin
      Result:=False;
      exit;
    end;

  Move(Data,FBuffer^,Bytes);
  Result:=True;
end;

function TRawBuffer.Fetch(var Container; Bytes: Integer): Integer;
begin
  if (not FAllocated) then
    begin
      Result:=0;
      exit;
    end;

  if (Bytes>FSize) then Bytes:=FSize;
  Move(FBuffer^,Container,Bytes);
  Result:=Bytes;
end;

function TRawBuffer.Copy: TRawBuffer;
var NewBuf: TRawBuffer;
begin
  if (not FAllocated) then
    begin
      NewBuf:=TRawBuffer.Create(FPeer);
      {$IFDEF RAWBUF_DEBUG}NewBuf.SetID(FID+'.Copy');{$ENDIF}
      Result:=NewBuf;
      exit;
    end;

  NewBuf:=TRawBuffer.Create(FSize,FPeer);
  {$IFDEF RAWBUF_DEBUG}NewBuf.SetID(FID+'.Copy');{$ENDIF}
  NewBuf.Store(FBuffer^,FSize);
  Result:=NewBuf;
end;

function TRawBuffer.MigrateTo(var NewBuffer: TRawBuffer): Boolean;
begin
  Result:=NewBuffer.ReceiveMigration(FBuffer,FSize);

  FAllocated:=False;
  FBuffer:=nil;
  FSize:=0;
end;

function TRawBuffer.ReceiveMigration(Buffer: Pointer; Size: Integer): boolean;
begin
  Deallocate;

  FAllocated:=(Size>0) and Assigned(Buffer);
  if (FAllocated) then
    begin
      FBuffer:=Buffer;
      FSize:=Size;
    end;
  Result:=FAllocated;
end;

function TRawBuffer.Append(var Buffer: TRawBuffer): Boolean;
var
  TmpBuffer: Pointer;
  NewSize: Integer;
  AppendPosition: Pointer;
begin
  Result:=False;

  // allocate enough memory for the combined buffers
  NewSize:=Buffer.Size+FSize;
  GetMem(TmpBuffer,NewSize);

  // figure out where in the temporary buffer the new buffer should be placed
  AppendPosition:=Pointer( Integer(TmpBuffer) + FSize );

  // move over our existing buffer
  Move(FBuffer^,TmpBuffer^,FSize);
  // move over the new buffer
  Move(Buffer.Buffer^,AppendPosition^,Buffer.Size);

  // the rest of the process is identical to receiving a migration, so we let
  // the ReceiveMigration method take over to deallocate our existing buffer
  // and put the new one in place
  Result:=ReceiveMigration(TmpBuffer,NewSize);
end;

procedure TRawBuffer.RemoveLeadingBytes(Bytes: Integer);
var
  TmpBuffer: Pointer;
  NewSize: Integer;
  StartPosition: Pointer;
begin
  // figure out how many bytes should be left, and allocate an appropriately-sized
  // buffer
  NewSize:=FSize-Bytes;
  GetMem(TmpBuffer,NewSize);

  // figure out where the trimmed data starts, and move it into the temporary buffer
  StartPosition:=Pointer( Integer(FBuffer) + Bytes );
  Move(StartPosition^,TmpBuffer^,NewSize);

  // the rest of the process is identical to receiving a migration, so we let
  // the ReceiveMigration method take over to deallocate our existing buffer
  // and put the new one in place
  ReceiveMigration(TmpBuffer,NewSize);
end;


procedure TRawBuffer.ResetCursor;
begin
  FCursor:=0;
end;

function TRawBuffer.StoreAtCursor(var Data; Bytes: Integer): boolean;
var Destination: Pointer;
begin
  Result:=False;

  // don't do the move if it would exceed our buffer size
  if FCursor+Bytes>FSize then exit;

  Destination:=Pointer( Integer(FBuffer) + FCursor );
  Move(Data,Destination^,Bytes);
  Inc(FCursor,Bytes);

  result:=true;
end;

function TRawBuffer.FetchAtCursor(var Container; Bytes: Integer): Integer;
var
  Source: Pointer;
begin
  // recalculate bytes to return only the rest of the buffer if the read
  // would otherwise exceed the buffer bounds 
  if FCursor+Bytes>FSize then Bytes:=FSize-FCursor;

  Source:=Pointer( Integer(FBuffer) + FCursor );
  Move(Source^,Container,Bytes);

  Inc(FCursor,Bytes);
  
  Result:=Bytes;
end;

{$IFDEF RAWBUF_DEBUG}
procedure TRawBuffer.SetID(ID: String);
begin
  FID:=ID;
  SendMessage(HWND(frmMain.Handle),WM_USER,MSG_RAWBUF_NAME,Integer(Self));
end;
{$ENDIF}

function TRawBuffer.StoreString(S: String): boolean;
begin
  Result:=Store(S[1],Length(S));
end;

function TRawBuffer.StoreStringAtCursor(S: String): boolean;
begin
  Result:=StoreAtCursor(S[1],Length(S));
end;


function TRawBuffer.AsString: String;
var
  S: String;
  i: Integer;
begin
  Result:='';
  if not FAllocated then exit;

  SetLength(S,FSize);
  Move(FBuffer^,S[1],FSize);

  for i:=1 to Length(S) do if S[i]=#0 then S[i]:=#32;

  Result:=S;
end;


end.
